use std::collections::{HashMap, HashSet};

use crate::Segment;

use super::Point;

fn indice_et_distance_point_interne_plus_eloigne(chemin: &[Point]) -> Option<(usize, f64)> {
    if chemin.len() < 3 {
        return None;
    }
    let debut = chemin.first().unwrap();
    let fin = chemin.last().unwrap();
    let raccourcis = Segment::new(*debut, *fin);
    chemin
        .iter()
        .map(|p| raccourcis.distance_au_point(p))
        .enumerate()
        .skip(1)
        .rev()
        .skip(1)
        .max_by(|(_, d1), (_, d2)| d1.partial_cmp(d2).unwrap())
}

pub fn simplification_gloutonne(chemin: &[Point], seuil: f64) -> Vec<Point> {
    if let Some((i, d)) = indice_et_distance_point_interne_plus_eloigne(chemin) {
        if d <= seuil {
            chemin
                .first()
                .into_iter()
                .chain(chemin.last())
                .copied()
                .collect()
        } else {
            let gauche = &chemin[0..=i];
            let droite = &chemin[i..];
            let mut gauche_simplifiee = simplification_gloutonne(gauche, seuil);
            let mut droite_simplifiee = simplification_gloutonne(droite, seuil);
            gauche_simplifiee.pop();
            gauche_simplifiee.append(&mut droite_simplifiee);
            gauche_simplifiee
        }
    } else {
        chemin.to_vec()
    }
}

fn choix_points(
    chemin: &[Point],
    seuil: f64,
    intervalle: (usize, usize),
    cache: &mut HashMap<(usize, usize), (Option<usize>, usize)>,
) -> usize {
    let (debut, fin) = intervalle;
    if debut == fin - 1 {
        2
    } else if let Some((_, valeur_opt)) = cache.get(&intervalle) {
        *valeur_opt
    } else {
        let point_debut = &chemin[debut];
        let point_fin = &chemin[fin];
        let raccourcis = Segment::new(*point_debut, *point_fin);
        let (choix_opt, valeur_opt) = if chemin[debut..fin]
            .iter()
            .all(|p| raccourcis.distance_au_point(p) <= seuil)
        {
            (None, 2)
        } else {
            (debut + 1..fin)
                .map(|milieu| {
                    (
                        Some(milieu),
                        choix_points(chemin, seuil, (debut, milieu), cache)
                            + choix_points(chemin, seuil, (milieu, fin), cache)
                            - 1,
                    )
                })
                .min_by_key(|(_, val)| *val)
                .unwrap()
        };
        cache.insert(intervalle, (choix_opt, valeur_opt));
        valeur_opt
    }
}

fn calcul_points_a_garder(
    cache: &HashMap<(usize, usize), (Option<usize>, usize)>,
    intervalle: (usize, usize),
    a_garder: &mut HashSet<usize>,
) {
    let (debut, fin) = intervalle;
    if let Some(milieu) = cache.get(&intervalle).and_then(|(o, _)| *o) {
        calcul_points_a_garder(cache, (debut, milieu), a_garder);
        calcul_points_a_garder(cache, (milieu, fin), a_garder);
        a_garder.insert(milieu);
    }
}

pub fn simplification_chemin(chemin: &[Point], seuil: f64) -> Vec<Point> {
    if chemin.len() <= 1 {
        return chemin.to_vec();
    }
    let mut cache = HashMap::new();
    let fin = chemin.len() - 1;
    let intervalle = (0, fin);
    choix_points(chemin, seuil, intervalle, &mut cache);
    let mut a_garder = HashSet::new();
    a_garder.insert(0);
    a_garder.insert(fin);
    calcul_points_a_garder(&cache, intervalle, &mut a_garder);
    chemin
        .iter()
        .enumerate()
        .filter_map(|(i, p)| a_garder.contains(&i).then_some(p))
        .copied()
        .collect()
}

pub fn simplification_hybride(chemin: &[Point], seuil: f64) -> Vec<Point> {
    if chemin.len() < 500 {
        simplification_chemin(chemin, seuil)
    } else {
        let (i, d) = indice_et_distance_point_interne_plus_eloigne(chemin).unwrap();
        if d <= seuil {
            chemin
                .first()
                .into_iter()
                .chain(chemin.last())
                .copied()
                .collect()
        } else {
            let (gauche, droite) = chemin.split_at(i);
            let mut gauche_simplifiee = simplification_hybride(gauche, seuil);
            let mut droite_simplifiee = simplification_hybride(droite, seuil);
            gauche_simplifiee.pop();
            gauche_simplifiee.append(&mut droite_simplifiee);
            gauche_simplifiee
        }
    }
}

pub fn simplification_hybride_parallele(chemin: &[Point], seuil: f64) -> Vec<Point> {
    if chemin.len() < 500 {
        simplification_chemin(chemin, seuil)
    } else {
        let (i, d) = indice_et_distance_point_interne_plus_eloigne(chemin).unwrap();
        if d <= seuil {
            chemin
                .first()
                .into_iter()
                .chain(chemin.last())
                .copied()
                .collect()
        } else {
            let gauche = &chemin[0..=i];
            let droite = &chemin[i..];
            let (mut gauche_simplifiee, mut droite_simplifiee) = rayon::join(
                || simplification_hybride_parallele(gauche, seuil),
                || simplification_hybride_parallele(droite, seuil),
            );
            gauche_simplifiee.pop();
            gauche_simplifiee.append(&mut droite_simplifiee);
            gauche_simplifiee
        }
    }
}
