use super::Point;
use xml::attribute::OwnedAttribute;

use gpx::{Gpx, read};
use std::{collections::HashMap, fs::File, io::BufReader, path::Path};

pub fn lecture_chemin<P: AsRef<Path>>(nom_fichier: P) -> std::io::Result<Vec<Point>> {
    let fichier_gpx = BufReader::new(File::open(nom_fichier)?);

    let gpx: Gpx = read(fichier_gpx)
        .map_err(|e| std::io::Error::new(std::io::ErrorKind::InvalidData, e.to_string()))?;

    let track = &gpx.tracks[0];

    Ok(track
        .segments
        .iter()
        .flat_map(|s| s.points.iter().map(|wp| wp.point()))
        .collect())
}

fn lecture_attribut<A: std::str::FromStr>(attributs: &[OwnedAttribute], clef: &str) -> Option<A> {
    for attribut in attributs {
        if attribut.name.local_name == clef {
            return attribut.value.parse::<A>().ok();
        }
    }
    None
}

pub fn lecture_xml(nom_fichier: &str) -> std::io::Result<Vec<Vec<Point>>> {
    let lecteur = std::io::BufReader::new(std::fs::File::open(nom_fichier)?);
    let lecteur_xml = xml::EventReader::new(lecteur);

    let mut noeuds: HashMap<i64, Point> = HashMap::new();
    let mut voie_courante: Vec<i64> = Vec::new();
    let mut voies = Vec::new();

    for evenement in lecteur_xml {
        let evenement = evenement
            .map_err(|e| std::io::Error::new(std::io::ErrorKind::InvalidData, e.to_string()))?;
        match evenement {
            xml::reader::XmlEvent::StartElement {
                name, attributes, ..
            } => {
                if name.local_name == "node" {
                    if let Some(id) = lecture_attribut(&attributes, "id")
                        && let Some(lat) = lecture_attribut(&attributes, "lat")
                        && let Some(lon) = lecture_attribut(&attributes, "lon")
                    {
                        noeuds.insert(id, Point::new(lon, lat));
                    }
                } else if name.local_name == "nd" {
                    voie_courante.extend(lecture_attribut::<i64>(&attributes, "ref"));
                }
            }
            xml::reader::XmlEvent::EndElement { name } if name.local_name == "way" => {
                let mut nouvelle_voie = Vec::new();
                std::mem::swap(&mut nouvelle_voie, &mut voie_courante);
                voies.push(nouvelle_voie);
            }
            _ => (),
        }
    }
    Ok(extraction_voies(voies, &noeuds))
}

fn extraction_voies(voies: Vec<Vec<i64>>, noeuds: &HashMap<i64, Point>) -> Vec<Vec<Point>> {
    voies
        .into_iter()
        .map(|voie| voie.iter().filter_map(|p| noeuds.get(p).copied()).collect())
        .collect()
}
