mod simplification;
use itertools::Itertools;
use simplification::{
    simplification_chemin, simplification_gloutonne, simplification_hybride,
    simplification_hybride_parallele,
};
mod svg;
use geo_types::Point;
use svg::{svg_carte_et_chemin, svg_chemin};

pub struct Segment {
    debut: Point,
    fin: Point,
}

impl Segment {
    fn new(p1: Point, p2: Point) -> Self {
        Segment { debut: p1, fin: p2 }
    }
    fn svg(&self) -> String {
        let (x1, y1) = self.debut.x_y();
        let (x2, y2) = self.fin.x_y();
        format!("<line x1='{x1}' y1='{y1}' x2='{x2}' y2='{y2}'/>")
    }
    fn delta(&self) -> [f64; 2] {
        [self.fin.x() - self.debut.x(), self.fin.y() - self.debut.y()]
    }
    fn longueur(&self) -> f64 {
        self.longueur_carree().sqrt()
    }
    fn longueur_carree(&self) -> f64 {
        let [diff_x, diff_y] = self.delta();
        diff_x * diff_x + diff_y * diff_y
    }
    fn distance_au_point(&self, p: &Point) -> f64 {
        let debut_vers_point = Segment::new(self.debut, *p);
        let l2 = self.longueur_carree();
        if l2 <= f64::EPSILON {
            return debut_vers_point.longueur();
        }
        // si on note d le debut du segment et f la fin
        // la line du segment a l'equation d + t (f - d).
        // pour trouver la projection de p sur la ligne
        // on a t = [(p-d) . (f-d)] / |f-d|^2
        // t est garde entre [0,1] pour rester dans le segment.
        let [x0, y0] = debut_vers_point.delta();
        let [x1, y1] = self.delta();
        let dot = x0 * x1 + y0 * y1;
        let t = (dot / l2).clamp(0.0, 1.0);

        let proj = Point::new(self.debut.x() + x1 * t, self.debut.y() + y1 * t);
        Segment::new(*p, proj).longueur()
    }
}

pub fn calcul_dimensions<'p, I: IntoIterator<Item = &'p Point>>(points: I) -> [f64; 4] {
    let mut xmin = f64::MAX;
    let mut ymin = f64::MAX;
    let mut xmax = f64::MIN;
    let mut ymax = f64::MIN;
    for (x, y) in points.into_iter().map(|p| (p.x(), p.y())) {
        if x < xmin {
            xmin = x
        }
        if x > xmax {
            xmax = x
        }
        if y < ymin {
            ymin = y
        }
        if y > ymax {
            ymax = y
        }
    }
    [xmin, ymin, xmax, ymax]
}

pub fn voie_vers_segments(voie: &[Point]) -> impl Iterator<Item = Segment> {
    voie.iter()
        .copied()
        .tuple_windows()
        .map(|(d, f)| Segment { debut: d, fin: f })
}

pub fn voies_vers_segments(voies: &[Vec<Point>]) -> impl Iterator<Item = Segment> {
    voies.iter().flat_map(|voie| voie_vers_segments(voie))
}

fn polygone(nombre_cotes: u32) -> Vec<Point> {
    let angle = std::f64::consts::PI * 2. / nombre_cotes as f64;
    (0..nombre_cotes)
        .map(|n| {
            let a = angle * n as f64;
            Point::new(a.cos(), a.sin())
        })
        .collect()
}

mod parseurs;
use parseurs::{lecture_chemin, lecture_xml};

fn main() -> std::io::Result<()> {
    let chemin = lecture_chemin("tour_ensimag.gpx")?;
    eprintln!("le chemin initial a {} points", chemin.len());
    let debut = std::time::Instant::now();
    let chemin = simplification_hybride_parallele(&chemin, 0.00005);
    eprintln!(
        "le chemin simplifie a {} points (calcule en {:?})",
        chemin.len(),
        debut.elapsed()
    );
    let carte = lecture_xml("interpreter")?;
    println!("{}", svg_carte_et_chemin(&chemin, &carte));
    Ok(())
}
